/*
 * If distributed as part of the Linux kernel, this code is licensed under the
 * terms of the GPL v2.
 *
 * Otherwise, the following license terms apply:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1) Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3) The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * common data structure and control functions for fcu and fcud
 */
#include <linux/kernel.h>
#include <linux/printk.h>
#include <linux/fs.h>		/* file or file_operation */
#include <linux/slab.h>		/* for kmalloc */
#include <linux/cdev.h>
#include <linux/wait.h>
#include <linux/device.h>	/* for class etc... */
#include <linux/spinlock.h>	/* for class etc... */
#include <linux/sysctl.h>	/* for sysctl etc... */
#include <linux/module.h>
#include "fcuvar.h"
#include "fax_share.h"
#include "fcucom.h"

#ifdef	DRIVER_DEBUG
static int __init init_sysctl(void);
static void __exit cleanup_sysctl(void);
#endif

static int fcuprobe(void);
 
static fax_share_param_t share_param;
static bool initdone = false;

static int __init  fax_init(void);
static void fax_exit(void);

extern int fcu_attach(void *);
extern int fcu_detach(void *);
extern int fcud_attach(void *);
extern int fcud_detach(void *);

extern int fcu_pci_runtime_suspend(struct device*);
extern int fcu_pci_runtime_resume(struct device*);

unsigned char isFcuExist = 0;
unsigned long isFcuActiveVaddr = 0;

// FAXボード装着確認用
ulong FAX_DETECT_REG = 0xD4019600;
ulong FAX_DETECT_BIT = 0x00000800;

// モデムボード電源確認用
ulong POK_FAX_REG = 0xD4019300;
ulong POK_FAX_BIT = 0x00002000;

// 共有エリアのベースアドレス
u_long FBI_COMMON_AREA_BASE = 0x3EFFFC00;

// A53とR4のメモリ境界アドレス
u_long A53_R4_BOUNDARY_ADDRESS = 0x7E700000;

module_param(FAX_DETECT_REG, ulong , 0664);
module_param(FAX_DETECT_BIT, ulong , 0664);
module_param(POK_FAX_REG, ulong , 0664);
module_param(POK_FAX_BIT, ulong , 0664);
module_param(FBI_COMMON_AREA_BASE, ulong , 0664);
module_param(A53_R4_BOUNDARY_ADDRESS, ulong, 0664);

/* inspired from NetBSD device cd structure */
extern struct file_operations fcu_FileOps;
extern struct file_operations fcud_FileOps;
extern struct fcu_softc g_fcu_softc;
extern struct fcud_softc g_fcud_softc;
struct devlist fax_devlist[2] = {
	/* minor = 0 : fcu */
	{
		.softc = (void*)(&g_fcu_softc),
		.ops = &fcu_FileOps,
		.name = "fcu",
		.attachfunc = fcu_attach,
		.detachfunc = fcu_detach,
	},
	/* minor = 1 : fcud */
	{
		.softc = (void*)(&g_fcud_softc),
		.ops = &fcud_FileOps,
		.name = "fcud",
		.attachfunc = fcud_attach,
		.detachfunc = fcud_detach,
	},
};
EXPORT_SYMBOL(fax_devlist);

static void init_fax_share_param(void)
{
	if( initdone == true ){
		return;
	}
	
	memset(&share_param, 0, sizeof(share_param));

	initdone = true;
}

fax_share_param_t * get_fax_share_param(void)
{
	if(initdone == false){
		init_fax_share_param();
	}
	return &share_param;
}
EXPORT_SYMBOL(get_fax_share_param);

/*
 * isFcuActive
 * FCUの動作状態を返す
 * 戻り値
 * FCU非動作中：FCUD_FCU_NOTACTIVE
 * FCU動作中：FCUD_FCU_ACTIVE
 */
int isFcuActive(void){
	unsigned long tmpReg;
	int ret = FCUD_FCU_NOTACTIVE;

	if(!isFcuExist){
		PRINT_INFO("Fcu Not Exist\n");
		ret = FCUD_FCU_NOTACTIVE;
	} else {
		if(!isFcuActiveVaddr){
			PRINT_INFO("POK_FAX_REG=%lx\n", POK_FAX_REG);
			PRINT_INFO("POK_FAX_BIT=%lx\n", POK_FAX_BIT);
			isFcuActiveVaddr = (unsigned long)ioremap_nocache(POK_FAX_REG, 16);
		}
		tmpReg = ioread32((void *)isFcuActiveVaddr);
		PRINT_INFO("ioread poxfax = %lx\n", tmpReg);
		PRINT_INFO("POK_FAX_BIT = %lx\n", POK_FAX_BIT);
		if(tmpReg & POK_FAX_BIT){
			PRINT_INFO("Fcu Active\n");
			ret = FCUD_FCU_ACTIVE;
		}else{
			PRINT_INFO("Fcu Not Active\n");
			ret = FCUD_FCU_NOTACTIVE;
		}
	}
	return ret;
}

/*
 * probe
 */
static int
fcuprobe() {
	int error = -1, major, minor;
	fax_driver_header_t *header;
	dev_t device;
	struct class *fax_class;
	struct device *dev;
	
	PRINT_INFO("%s: fcuprobe_start\n", DEV_NAME);
	
	/* memory initial */
	init_fax_share_param();

	/* allocate device class */
	fax_class = class_create(THIS_MODULE, DEV_NAME);
	if(IS_ERR(fax_class)){
		PRINT_ERR("%s: can't create class\n", DEV_NAME);
		error = PTR_ERR(fax_class);
		goto class_create_error;	
	}
	
	/* allocate device file,2 means fcu and fcud */
	error = alloc_chrdev_region(&device, 0, 2, DEV_NAME);
	if (error) {
		PRINT_ERR( "%s: can't alloc major%d, error=%d\n", DEV_NAME, MAJOR(device), error);
		goto chrdev_alloc_error;
	}
	/* driver specific initialization */
	for( minor = 0; minor < ARRAY_SIZE(fax_devlist); minor++ ){
		if (fax_devlist[minor].attachfunc(fax_devlist[minor].softc)!= 0){
			PRINT_ERR( "%s: attacherror minor=%d, error=%d\n", DEV_NAME, minor, error);
			goto attach_error;
		}
	}
	/* register minor device interface to char driver framework*/
	major = MAJOR(device);
	
	for( minor = 0; minor < ARRAY_SIZE(fax_devlist); minor++ ){
		/* header must be first member of each softc */
		header = fax_devlist[minor].softc;
		/* set major level device */
		header->sc_dev = MKDEV(major, minor);
		/* set class structure */
		header->sc_class = fax_class;
		header->sc_name = fax_devlist[minor].name;
		/* add character device to kernel */
		cdev_init(&header->sc_cdev, fax_devlist[minor].ops);
		header->sc_cdev.owner = THIS_MODULE;
		error = cdev_add(&header->sc_cdev, header->sc_dev, 1);
		if( error ){
			PRINT_ERR("%s: can't register cdev(%d)d, error=%d\n", DEV_NAME, MINOR(header->sc_dev), error);
			goto cdev_add_error;
		}
		dev = device_create(fax_class, NULL, header->sc_dev, NULL, fax_devlist[minor].name);
		if (IS_ERR(dev)){
			PRINT_ERR("%s: can't create cdev(%d)d\n", DEV_NAME, MINOR(header->sc_dev));
			goto device_create_error;
		}
	}
	PRINT_INFO("%s: fcuprobe_end\n", DEV_NAME);
	
	return 0;

device_create_error:
cdev_add_error:
	for( minor-- ; minor > 0; minor-- ){
		header = fax_devlist[minor].softc;
		device_destroy(header->sc_class, header->sc_dev);
		cdev_del(&header->sc_cdev);
	}
attach_error:
	unregister_chrdev_region(device, ARRAY_SIZE(fax_devlist));
chrdev_alloc_error:
	class_destroy(fax_class);
class_create_error:
	return error;
}

static int __init  fax_init(void)
{

	unsigned long vaddr, tmpReg;
	
	PRINT_INFO("%s: fax_init\n", DEV_NAME);
	PRINT_INFO("FAX_DETECT_REG=%lx\n", FAX_DETECT_REG);
	PRINT_INFO("FAX_DETECT_BIT=%lx\n", FAX_DETECT_BIT);
	PRINT_INFO("FBI_COMMON_AREA_BASE=%lx\n", FBI_COMMON_AREA_BASE);
	PRINT_INFO("A53_R4_BOUNDARY_ADDRESS=%lx\n", A53_R4_BOUNDARY_ADDRESS);
	vaddr = (unsigned long)ioremap_nocache(FAX_DETECT_REG, 16);
	tmpReg = ioread32((void *)vaddr);
	PRINT_INFO("ioread fcudetect = %lx\n", tmpReg);
	
	isFcuExist = !(tmpReg & FAX_DETECT_BIT); // 0でFAXボード装着、1でFAXボード未装着
	if(isFcuExist){
		fcuprobe();
	}
	return 0;
}
EXPORT_SYMBOL(fax_init);


static void fax_exit(void)
{
	int minor;
	fax_driver_header_t *header;

	PRINT_INFO("%s: fcuExit\n", DEV_NAME);

	for( minor = 0; minor < ARRAY_SIZE(fax_devlist); minor++ ){
		header = fax_devlist[minor].softc;
		device_destroy(header->sc_class, header->sc_dev);
		cdev_del(&header->sc_cdev);
	}
	unregister_chrdev_region(header->sc_dev, ARRAY_SIZE(fax_devlist));
	class_destroy(header->sc_class);
	
	for( minor = 0; minor < ARRAY_SIZE(fax_devlist); minor++ ){
		fax_devlist[minor].detachfunc(fax_devlist[minor].softc);
	}
}
EXPORT_SYMBOL(fax_exit);

module_init(fax_init);
module_exit(fax_exit);

MODULE_AUTHOR("RICOH Company, LTD.");
MODULE_LICENSE("GPL");


